// $Id: CDriveInformation.cpp,v 1.2 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CDriveInformation.hpp"
using Exponent::Host::CDriveInformation;

#ifdef WIN32
//	===========================================================================
const char *CDriveInformation::CDRIVE_INFO_DRIVE_STRINGS[] =
{
	"Unknown Drive Type",
	"Unmounted Drive",
	"Removable Drive",
	"Fixed Drive",
	"Network Drive",
	"CD ROM Drive",
	"RAM Drive"
};
#endif

//	===========================================================================
void CDriveInformation::dumpDriveInfoToFile(const CSystemString &path)
{
	// Copy the path
	CSystemString myPath = path;

	// Check that the path is correct..
	if (!myPath.hasExtension())
	{
		myPath.appendString("CPUStats.txt");
	}

	// Create the stream
	CTextStream stream(myPath, CTextStream::e_output);

	// Output
	dumpDriveInfoToFile(stream);

	// Close the file
	stream.closeStream();
}

//	===========================================================================
void CDriveInformation::dumpDriveInfoToFile(CTextStream &stream)
{
	// Check if the stream is open
	if (stream.isStreamOpen())
	{
		// Write the title
		stream << "Exponent Host drive info extraction tool\n";
		stream << "----------------------------------------\n\n";

		// If windows detail secifics
#ifdef WIN32
		CSystemString sys;
		getWindowsFolder(sys);
		stream << "Windows folder : " << sys << "\n";

		getWindowsTemporaryFolder(sys);
		stream << "Window temp folder : " << sys << "\n\n";
		stream << "----------------------------------------\n\n";
#endif

		// Get the drive strings
		TStringCountedPointerArray array;
		getDriveNames(array);

		for (long i = 0; i < array.getArraySize(); i++)
		{
			// First output the device handles
			CString *str = array.elementAtIndex(i);

			// Check its a valid stirng
			if (str == NULL)
			{
				continue;
			}

			// Output the registered name
			stream << "Registered name : " << *str << "\n";

			// Now we need to output info about the drive
			CString volumeName;
			CString fileSystem;
			long serial;
			getDriveInformation(*str, volumeName, fileSystem, serial);

			stream << "Volume name : " << volumeName << "\n";
			stream << "File system : " << fileSystem << "\n";
			stream << "Serial code : " << serial << "\n";

			// Volume type
	#ifdef WIN32
			stream << "Volume is of type : " << CDRIVE_INFO_DRIVE_STRINGS[getDriveType(*str)] << "\n";
	#endif
		stream << "----------------------------------------\n\n";
		}

		// Close the stream to finish
		stream.closeStream();
	}
}

//	===========================================================================
void CDriveInformation::getDriveNames(TStringCountedPointerArray &array)
{
#ifdef WIN32
	char buffer[2048];
	buffer[0] = 0;
	buffer[1] = 0;

	GetLogicalDriveStrings(2048, buffer);

	// Now copy out the strings
	char *bufferPtr = buffer;

	while(*bufferPtr != 0)
	{
		array.addElement(new CString(bufferPtr));
		while(*bufferPtr++ != 0) { /* Loop till we find next valid string */ }
	}
#else
	FSVolumeRefNum refNum;
	HFSUniStr255 unicodeString;
	OSErr error;
	long i = 1;
	while ((error = FSGetVolumeInfo (kFSInvalidVolumeRefNum, i, &refNum, kFSVolInfoNone, NULL, &unicodeString, NULL)) == noErr)
	{
		CFStringRef string = CFStringCreateWithCharacters(kCFAllocatorDefault, unicodeString.unicode, unicodeString.length);
		char buffer[1024];
		CFStringGetCString (string, buffer, 1024 * sizeof(char), kCFStringEncodingMacRoman);
		array.addElement(new CString(buffer));
		CFRelease(string);
		i++;
	}
#endif
}

//	===========================================================================
void CDriveInformation::getDriveInformation(const CString &driveToExamine, CString &volumeName, CString &fileSystem, long &volumeSerialNumber)
{
#ifdef WIN32
	char volName[512];		// Storage for the volume name
	char fileSys[512];		// Storage for the file system
	DWORD serial;			// Storage for the serial number

	if (GetVolumeInformation(driveToExamine.getString(), volName, 512, (DWORD *)&serial, 0, 0, fileSys, 512) == TRUE)
	{
		volumeName = volName;
		fileSystem = fileSys;
		volumeSerialNumber = serial;
	}
#else
	FSVolumeRefNum refNum;
	HFSUniStr255 unicodeString;
	OSErr error;
	long i = 1;
	FSVolumeInfo info;
	memset(&info, 0, sizeof(info));
	while ((error = FSGetVolumeInfo (kFSInvalidVolumeRefNum, i, &refNum, kFSVolInfoDriveInfo, &info, &unicodeString, NULL)) == noErr)
	{
		CFStringRef string = CFStringCreateWithCharacters(kCFAllocatorDefault, unicodeString.unicode, unicodeString.length);
		char buffer[1024];
		CFStringGetCString (string, buffer, 1024 * sizeof(char), kCFStringEncodingMacRoman);
		CFRelease(string);
		CString stringBuffer = buffer;
		if (stringBuffer == driveToExamine)
		{
			// Found the matching string
			fileSystem = "MAC OSX";
			volumeName = stringBuffer;
			volumeSerialNumber = info.driveNumber;
			return;
		}
		i++;
	}

#endif
}

#ifdef WIN32
//	===========================================================================
long CDriveInformation::getDriveType(const CString &driveToExamine)
{
	return GetDriveType(driveToExamine.getString());
}

//	===========================================================================
bool CDriveInformation::driveIsCdrom(const CString &driveToExamine)
{
	return (getDriveType(driveToExamine) == CDRIVE_INFO_CDROM_DRIVE);
}

//	===========================================================================
bool CDriveInformation::driveIsNetworkDrive(const CString &driveToExamine)
{
	return (getDriveType(driveToExamine) == CDRIVE_INFO_NETWORK_DRIVE);
}

//	===========================================================================
bool CDriveInformation::driveIsFixedDrive(const CString &driveToExamine)
{
	return (getDriveType(driveToExamine) == CDRIVE_INFO_FIXED_DRIVE);
}

//	===========================================================================
void CDriveInformation::getDriveTypeAsString(const CString &driveToExamine, CString type)
{
	type = CDRIVE_INFO_DRIVE_STRINGS[getDriveType(driveToExamine)];
}

//	===========================================================================
void CDriveInformation::getWindowsTemporaryFolder(CSystemString &folder)
{
	char tempFolder[2048];
	memset(tempFolder, 0, 2048 * sizeof(char));
	GetTempPath(2048, tempFolder);
	folder = tempFolder;
}

//	===========================================================================
void CDriveInformation::getWindowsFolder(CSystemString &folder)
{
	char winFolder[2048];
	memset(winFolder, 0, 2048 * sizeof(char));
	GetWindowsDirectory(winFolder, 2048);
	folder = winFolder;
}
#endif